#if NUNIT

#region References

using System;
using System.Data;
using NUnit.Framework;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7.Parsers;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using gov.va.med.vbecs.DAL.HL7AL;

using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

using gov.va.med.vbecs.UnitTests;

#endregion

namespace gov.va.med.vbecs.DAL.HL7.UnitTests.Parsers
{

	[TestFixture]
	public class Parsers_CprsNewOrder : BaseTest 
	{
		private const string RBC_COMPONENT_ORDER = "2^RED BLOOD CELLS^99VBC^^";
		private const string FFP_COMPONENT_ORDER = "3^FRESH FROZEN PLASMA^99VBC^^";
		private const string CRYO_COMPONENT_ORDER = "5^CRYOPRECIPITATE^99VBC^^";
		private const string PLT_COMPONENT_ORDER = "4^PLATELETS^99VBC^^";
		private const string OTHER_COMPONENT_ORDER = "6^OTHER^99VBC^^";
		private const string WB_COMPONENT_ORDER = "11^WHOLE BLOOD^99VBC^^";
		private const string ABO_RH_TEST_ORDER = "7^ABO/RH^99VBC^^";
		private const string ABS_TEST_ORDER = "8^ABS^99VBC^^";
		private const string DAT_TEST_ORDER = "9^DAT^99VBC^^";
		private const string TAS_TEST_ORDER = "1^TAS^99VBC^^";
		private const string TRW_TEST_ORDER = "10^TRW^99VBC^^";
		private const string KC_DIVISION = "589^VA HEARTLAND - WEST, VISN 15";
		private const string WI_DIVISION = "589A7^WICHITA VAMC";
		private HL7Interface intParms;

		[SetUp]
		public void SetUp()
		{
			intParms = new HL7Interface( InterfaceName.CPRS.ToString() );
		}

        private int GetUniqueCprsOrderNumber()
        {
            int checkNum = System.Convert.ToInt32(SprocHelper.GetNewCprsOrderNumber());
          
            //Lohse - I dont want this to go FOREVER Askew!!
            bool found = false;
            for (int x = 0; x < 100; x++)
            {
                DataTable dtTest = UnitTestUtil.RunSQLGetDataTable("SELECT 1 FROM OrderedTest WHERE CprsOrderNumber = '" + checkNum.ToString() + "'");
                DataTable dtOrder = UnitTestUtil.RunSQLGetDataTable("SELECT 1 FROM PatientOrder WHERE OrderGroupNumber = '" + (checkNum-1).ToString() + "'");
                if (dtTest.Rows.Count + dtOrder.Rows.Count == 0)
                {
                    found = true;
                    break;
                }
                else
                {
                    checkNum = System.Convert.ToInt32(SprocHelper.GetNewCprsOrderNumber());
                }
            }

            return (found) ? checkNum :int.MinValue;
        }

	    [Test]
		public void CprsNewOrder_Pass_1()
	    {
	        int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

			int groupNum = cprsOrderNumber - 1;

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), TAS_TEST_ORDER, KC_DIVISION, HL7UnitTestUtil.GetRandomNumber(999999).ToString() );
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );

			Assert.IsNotNull( new CprsNewOrder( omgMessage, intParms ) );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void CprsNewOrder_Fail_1()
		{
			CprsNewOrder newOrder = new CprsNewOrder( null, intParms );
		}

		[Test]
		public void ProcessNewOrder_Pass_1()
		{
	        int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

			int groupNum = cprsOrderNumber - 1;

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), TAS_TEST_ORDER, KC_DIVISION, HL7UnitTestUtil.GetRandomNumber(999999).ToString() );
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, intParms ).ProcessNewOrder();
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMsg.Message ) , "Ack Check" );
		}

		[Test]
		[ExpectedException( typeof( HL7Exception ) )]
		public void ProcessNewOrder_Fail_1()
		{
			int cprsOrderNumber = HL7UnitTestUtil.GetRandomNumber(9999999);
			int groupNum = cprsOrderNumber - 1;
			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), TAS_TEST_ORDER, KC_DIVISION, HL7UnitTestUtil.GetRandomNumber(999999).ToString() );
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OmgMessage newMsg = new HL7OmgMessage( messageString.Replace( "ORC|NW", "ORC|NN" ) );
			HL7OrgMessage ackMsg = new CprsNewOrder( newMsg, intParms ).ProcessNewOrder();
		}

		[Test]
		public void ProcessNewOrder_Pass_2()
		{
            int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

            int groupNum = cprsOrderNumber - 1;
            DataRow dr = HL7UnitTestUtil.RunSQLGetDataRow("SELECT Top 1 VistaPatientId FROM Patient");

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), TAS_TEST_ORDER, KC_DIVISION, dr["VistaPatientId"].ToString());
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, intParms ).ProcessNewOrder();
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMsg.Message ) , "Ack Check" );
		}

		[Test]
		public void ProcessNewOrder_Pass_3()
		{
            int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

            int groupNum = cprsOrderNumber - 1;
            DataRow dr = HL7UnitTestUtil.RunSQLGetDataRow("SELECT Top 1 VistaPatientId FROM Patient");

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), ABO_RH_TEST_ORDER, KC_DIVISION, dr["VistaPatientId"].ToString());
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, intParms ).ProcessNewOrder();
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMsg.Message ) , "Ack Check" );
		}

		[Test]
		public void ProcessNewOrder_Pass_4()
		{
            int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

            int groupNum = cprsOrderNumber - 1;
            DataRow dr = HL7UnitTestUtil.RunSQLGetDataRow("SELECT Top 1 VistaPatientId FROM Patient");

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), ABS_TEST_ORDER, KC_DIVISION, dr["VistaPatientId"].ToString());
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, intParms ).ProcessNewOrder();
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMsg.Message ) , "Ack Check" );
		}

		[Test]
		public void ProcessNewOrder_Pass_5()
		{
            int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

            int groupNum = cprsOrderNumber - 1;
            DataRow dr = HL7UnitTestUtil.RunSQLGetDataRow("SELECT Top 1 VistaPatientId FROM Patient");

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), DAT_TEST_ORDER, KC_DIVISION, dr["VistaPatientId"].ToString());
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, intParms ).ProcessNewOrder();
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMsg.Message ) , "Ack CHeck" );
		}

		[Test]
		public void ProcessNewOrder_Pass_6()
		{
            int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

            int groupNum = cprsOrderNumber - 1;
            DataRow dr = HL7UnitTestUtil.RunSQLGetDataRow("SELECT Top 1 VistaPatientId FROM Patient");

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), TRW_TEST_ORDER, KC_DIVISION, dr["VistaPatientId"].ToString());
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, intParms ).ProcessNewOrder();
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMsg.Message ) , "Ack Check" );
		}

		[Test]
		public void ProcessNewOrder_Pass_7()
		{
            int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

            int groupNum = cprsOrderNumber - 1;
            DataRow dr = HL7UnitTestUtil.RunSQLGetDataRow("SELECT Top 1 VistaPatientId FROM Patient");

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), RBC_COMPONENT_ORDER, KC_DIVISION, dr["VistaPatientId"].ToString());
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, intParms ).ProcessNewOrder();
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMsg.Message ) , "Ack Check" );
		}

		[Test]
		public void ProcessNewOrder_Pass_8()
		{
            int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

            int groupNum = cprsOrderNumber - 1;
            DataRow dr = HL7UnitTestUtil.RunSQLGetDataRow("SELECT Top 1 VistaPatientId FROM Patient");

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), FFP_COMPONENT_ORDER, KC_DIVISION, dr["VistaPatientId"].ToString());
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, intParms ).ProcessNewOrder();
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMsg.Message ) , "Ack Check" );
		}

		[Test]
		public void ProcessNewOrder_Pass_9()
		{
            int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

            int groupNum = cprsOrderNumber - 1;
            DataRow dr = HL7UnitTestUtil.RunSQLGetDataRow("SELECT Top 1 VistaPatientId FROM Patient");

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), CRYO_COMPONENT_ORDER, KC_DIVISION, dr["VistaPatientId"].ToString());
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, intParms ).ProcessNewOrder();
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMsg.Message ) , "Ack CHeck" );
		}

		[Test]
		public void ProcessNewOrder_Pass_10()
		{
            int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

            int groupNum = cprsOrderNumber - 1;
            DataRow dr = HL7UnitTestUtil.RunSQLGetDataRow("SELECT Top 1 VistaPatientId FROM Patient");

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), PLT_COMPONENT_ORDER, KC_DIVISION, dr["VistaPatientId"].ToString());
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, intParms ).ProcessNewOrder();
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMsg.Message ) , "Ack CHeck" );
		}

		[Test]
		public void ProcessNewOrder_Pass_11()
		{
            int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

            int groupNum = cprsOrderNumber - 1;
            DataRow dr = HL7UnitTestUtil.RunSQLGetDataRow("SELECT Top 1 VistaPatientId FROM Patient");

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), WB_COMPONENT_ORDER, KC_DIVISION, dr["VistaPatientId"].ToString());
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, intParms ).ProcessNewOrder();
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMsg.Message ) , "Ack Check" );
		}

		[Test]
		public void ProcessNewOrder_Pass_12()
		{
            int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

            int groupNum = cprsOrderNumber - 1;
            DataRow dr = HL7UnitTestUtil.RunSQLGetDataRow("SELECT Top 1 VistaPatientId FROM Patient");

			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), OTHER_COMPONENT_ORDER, KC_DIVISION, dr["VistaPatientId"].ToString());
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
			HL7OrgMessage ackMsg = new CprsNewOrder( omgMessage, intParms ).ProcessNewOrder();
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMsg.Message ) , "Ack CHeck" );

		}

		[Test]
		public void CprsNewOrder_Static_Pass()
		{
			DeleteAssociatedInstitutions();
			//
			InsertAssociatedInstitutions();
			//
            int cprsOrderNumber = this.GetUniqueCprsOrderNumber();
            if (cprsOrderNumber == int.MinValue)
                Assert.Fail("Could not get unique CprsOrderNumber and OrderGroupNumber");

            int groupNum = cprsOrderNumber - 1;
			
			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", cprsOrderNumber.ToString(), groupNum.ToString(), TAS_TEST_ORDER, KC_DIVISION, HL7UnitTestUtil.GetRandomNumber(999999).ToString() );
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );

			Assert.IsNotNull( new CprsNewOrder( omgMessage, intParms ) );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.Untestable)]
		public void CprsNewOrder_Static_Fail()
		{

			// ****************************************************************************************************************************
			// *************************************************************************************
			// NOTE: For CprsNewOrder_Static_Pass() and CprsNewOrder_Static_Fail, because of the 
			// static Hashtable that holds AssociatedInstitutions we cannot run both tests
			// sequentially; the Unit Test App must be re-started to run each one - remove
			// the comments to run a test
			// *************************************************************************************

//			DeleteAssociatedInstitutions();
//			//
//			int orderNum = HL7UnitTestUtil.GetRandomNumber(9999999);
//			int groupNum = orderNum - 1;
//			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", orderNum.ToString(), groupNum.ToString(), TAS_TEST_ORDER, KC_DIVISION, HL7UnitTestUtil.GetRandomNumber(999999).ToString() );
//			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );
//
//			Assert.IsNotNull( new CprsNewOrder( omgMessage, intParms ) );
		}

		// ****************************************************************************************************************************

		#region Helper Methods

		/// <summary>
		/// 
		/// </summary>
		private void InsertAssociatedInstitutions()
		{
			DataTable dt = new DataTable(TABLE.AssociatedInstitution.TableName);

			dt.Columns.Add(TABLE.AssociatedInstitution.AssociatedInstitutionGuid, typeof(System.Guid));
			dt.Columns.Add(TABLE.AssociatedInstitution.DivisionCode, typeof(string));
			dt.Columns.Add(TABLE.AssociatedInstitution.InstitutionDivisionCode, typeof(string));
			dt.Columns.Add(TABLE.AssociatedInstitution.LastUpdateDate, typeof(string));
			dt.Columns.Add(TABLE.AssociatedInstitution.LastUpdateUser, typeof(string));
			dt.Columns.Add(TABLE.AssociatedInstitution.LastUpdateFunctionId, typeof(int));
			dt.Columns.Add(TABLE.AssociatedInstitution.RecordStatusCode, typeof(char));

			#region Add Data

			DataRow dr = dt.NewRow();
			dr[TABLE.AssociatedInstitution.AssociatedInstitutionGuid] = Guid.NewGuid();
			dr[TABLE.AssociatedInstitution.DivisionCode] = "589";
			dr[TABLE.AssociatedInstitution.InstitutionDivisionCode] = "589";
			dr[TABLE.AssociatedInstitution.LastUpdateDate] = DateTime.UtcNow;
			dr[TABLE.AssociatedInstitution.LastUpdateUser] = "TESTUSER";
			dr[TABLE.AssociatedInstitution.LastUpdateFunctionId] = (int)Common.UpdateFunction.UnitTests;
			dr[TABLE.AssociatedInstitution.RecordStatusCode] = 'A';
			dt.Rows.Add(dr);

			dr = dt.NewRow();
			dr[TABLE.AssociatedInstitution.AssociatedInstitutionGuid] = Guid.NewGuid();
			dr[TABLE.AssociatedInstitution.DivisionCode] = "589";
			dr[TABLE.AssociatedInstitution.InstitutionDivisionCode] = "INST1";
			dr[TABLE.AssociatedInstitution.LastUpdateDate] = DateTime.UtcNow;
			dr[TABLE.AssociatedInstitution.LastUpdateUser] = "TESTUSER";
			dr[TABLE.AssociatedInstitution.LastUpdateFunctionId] = (int)Common.UpdateFunction.UnitTests;
			dr[TABLE.AssociatedInstitution.RecordStatusCode] = 'A';
			dt.Rows.Add(dr);

			dr = dt.NewRow();
			dr[TABLE.AssociatedInstitution.AssociatedInstitutionGuid] = Guid.NewGuid();
			dr[TABLE.AssociatedInstitution.DivisionCode] = "589";
			dr[TABLE.AssociatedInstitution.InstitutionDivisionCode] = "INST2";
			dr[TABLE.AssociatedInstitution.LastUpdateDate] = DateTime.UtcNow;
			dr[TABLE.AssociatedInstitution.LastUpdateUser] = "TESTUSER";
			dr[TABLE.AssociatedInstitution.LastUpdateFunctionId] = (int)Common.UpdateFunction.UnitTests;
			dr[TABLE.AssociatedInstitution.RecordStatusCode] = 'A';
			dt.Rows.Add(dr);

			dr = dt.NewRow();
			dr[TABLE.AssociatedInstitution.AssociatedInstitutionGuid] = Guid.NewGuid();
			dr[TABLE.AssociatedInstitution.DivisionCode] = "589";
			dr[TABLE.AssociatedInstitution.InstitutionDivisionCode] = "INST3";
			dr[TABLE.AssociatedInstitution.LastUpdateDate] = DateTime.UtcNow;
			dr[TABLE.AssociatedInstitution.LastUpdateUser] = "TESTUSER";
			dr[TABLE.AssociatedInstitution.LastUpdateFunctionId] = (int)Common.UpdateFunction.UnitTests;
			dr[TABLE.AssociatedInstitution.RecordStatusCode] = 'A';
			dt.Rows.Add(dr);

			dr = dt.NewRow();
			dr[TABLE.AssociatedInstitution.AssociatedInstitutionGuid] = Guid.NewGuid();
			dr[TABLE.AssociatedInstitution.DivisionCode] = "589";
			dr[TABLE.AssociatedInstitution.InstitutionDivisionCode] = "INST4";
			dr[TABLE.AssociatedInstitution.LastUpdateDate] = DateTime.UtcNow;
			dr[TABLE.AssociatedInstitution.LastUpdateUser] = "TESTUSER";
			dr[TABLE.AssociatedInstitution.LastUpdateFunctionId] = (int)Common.UpdateFunction.UnitTests;
			dr[TABLE.AssociatedInstitution.RecordStatusCode] = 'A';
			dt.Rows.Add(dr);

			dr = dt.NewRow();
			dr[TABLE.AssociatedInstitution.AssociatedInstitutionGuid] = Guid.NewGuid();
			dr[TABLE.AssociatedInstitution.DivisionCode] = "589";
			dr[TABLE.AssociatedInstitution.InstitutionDivisionCode] = "INST5";
			dr[TABLE.AssociatedInstitution.LastUpdateDate] = DateTime.UtcNow;
			dr[TABLE.AssociatedInstitution.LastUpdateUser] = "TESTUSER";
			dr[TABLE.AssociatedInstitution.LastUpdateFunctionId] = (int)Common.UpdateFunction.UnitTests;
			dr[TABLE.AssociatedInstitution.RecordStatusCode] = 'A';
			dt.Rows.Add(dr);

			#endregion
			
			new Common.StoredProcedure().TransactionalGetValue( STOREDPROC.InsertAssociatedInstitution.StoredProcName, dt );
		}

		/// <summary>
		/// 
		/// </summary>
		private void DeleteAssociatedInstitutions()
		{
			HL7UnitTestUtil.RunSQL("DELETE FROM AssociatedInstitution");
		}

		#endregion
	}
}
#endif